<?php

class microsoft_oauth_client
{
    private string $client_id;
    private string $client_secret;

    public function __construct($client_id, $client_secret)
    {
        $this->client_id = $client_id;
        $this->client_secret = $client_secret;
    }

    /**
     * @param string $oauth_code
     * @param string $redirect_uri
     * @return mixed
     * @throws Exception
     */
    public function get_access_token(string $oauth_code, string $redirect_uri): mixed
    {
        $api_url = 'https://login.microsoftonline.com/common/oauth2/v2.0/token';

        $headers = [
            'content-type' => 'application/x-www-form-urlencoded'
        ];

        $body_data = http_build_query([
            'client_id' => $this->client_id,
            'client_secret' => $this->client_secret,
            'code' => $oauth_code,
            'scope' => 'https://graph.microsoft.com/.default',
            'redirect_uri' => $redirect_uri,
            'grant_type' => 'authorization_code',
        ]);

        $response = json_decode($this->curl_request($api_url, 'POST', $headers, $body_data));

        return $response->access_token;
    }

    /**
     * @param string $access_token
     * @return mixed
     * @throws Exception
     */
    public function get_authenticated_user(string $access_token): mixed
    {
        $api_url = 'https://graph.microsoft.com/v1.0/me';

        return json_decode($this->curl_request($api_url, 'GET',
            [
                'Authorization' => sprintf('Bearer %s', $access_token),
            ]
        ));

    }

    /**
     * @param string $url
     * @param string $method
     * @param array $headers
     * @param array|string $body
     * @param array $curl_options
     * @return bool|string
     * @throws Exception
     */
    private function curl_request(string $url, string $method = 'GET', array $headers = [], array|string $body = [], array $curl_options = []): bool|string
    {
        $curl = curl_init();

        array_change_key_case($headers, CASE_LOWER);

        $headers = array_merge(['accept' => 'application/json'], $headers);

        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_TIMEOUT, 0);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        if (str_starts_with($url, 'https://')) {
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
        }

        // If any headers are set, add them to curl request
        if (!empty($headers)) {
            curl_setopt($curl, CURLOPT_HTTPHEADER, array_map(function ($key, $value) {
                return $key . ': ' . $value;
            }, array_keys($headers), array_values($headers)));
        }

        // Set the request type , GET, POST, PUT or DELETE
        switch (strtoupper($method)) {
            case 'POST':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'POST');
                break;
            case 'PUT':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'PUT');
                break;
            case 'DELETE':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'DELETE');
                break;
            default:
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'GET');
                break;
        }

        // Add content/body data to send with request
        if (!empty($body)) {
            curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
        }

        // Any extra curl options to add to curl object
        if (!empty($curl_options)) {
            foreach ($curl_options as $option_key => $option_value) {
                curl_setopt($curl, $option_key, $option_value);
            }
        }

        $response = curl_exec($curl);

        $error = curl_error($curl);

        $error_code = curl_errno($curl);

        $status_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        if ($error_code > 0) {
            throw new Exception($error, $error_code);
        }

        if ($status_code !== 200) {
            throw new Exception($response, $status_code);
        }

        curl_close($curl);

        return $response;
    }
}